/* $Id: chanswap.c,v 1.10 1999/01/26 19:45:08 ericb Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* In the past, we have run into problems where data from different
   pairs of channels would get swapped.  Typically this was due to
   synchronization problems with the ADCs on the SCAs, so there is no
   way to detect it other than comparing measured data with what is
   known to be hooked up.  This program does some rudimentary tests to
   verify that data is where we expect it.

   Another problem we ran into in the past was with frequency data.
   If some channels in the middle were inactive, the frequency data
   after those channels would get messed up.  When the "-f"
   command-line option is given, this program will catch that as
   well.

   Yet another problem we ran into was when doing an e1432_preset,
   followed immediately enabling a bunch of channels and an
   e1432_init_measure.  This could cause channel swap on an E1432.
   This program will catch that problem as well.

   Yet another problem was with e1432_preset followed immediately by
   e1432_init_measure, where we could end up not waiting quite long
   enough for ADC settling and get bogus data.  This program will
   catch that problem as well. */

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"

#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_INPUT_CHANS)
#define	BLOCKSIZE	1024
#define	RANGE		5	/* Minimum 2V because we use 1.5V signals */
#define	SCANS_PER_MEAS	4
#define	LOOPS		8

/* #define	PLOT */
#ifdef	PLOT
#include <X11/Xlib.h>		/* For XOpenDisplay */
#include "../demo/xplot.h"	/* For xplot_init_plot */
/* Pixel spacing between windows */
#define	X_SPACING	15
#define	Y_SPACING	40
char   *plotID[NCHAN_MAX];
FLOATSIZ32 buffer[BLOCKSIZE * NCHAN_MAX];
#endif

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: chanswap.c,v 1.10 1999/01/26 19:45:08 ericb Exp $";
static const char *progname;

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].input_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

/*ARGSUSED*/
static int
setup(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
      int fft, double clock_freq, double span)
{
    CHECK(e1432_set_clock_freq(hw, group, clock_freq));
    CHECK(e1432_set_span(hw, group, span));
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    CHECK(e1432_set_append_status(hw, group, E1432_APPEND_STATUS_ON));
    CHECK(e1432_set_range(hw, group, RANGE));
    if (fft)
	CHECK(e1432_set_calc_data(hw, group, E1432_DATA_FREQ));

    return 0;
}

/*ARGSUSED*/
static int
setup1(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
       int fft, int active[][NCHAN_MAX])
{
    int type, chan;

    /* Initialize active array */
    for (type = 0; type < 2; type++)
	for (chan = 0; chan < nchan; chan++)
	{
	    if (type > 0 && !fft)
		active[type][chan] = 0;
	    else
		active[type][chan] = 1;
	}

    /* Deactivate first two channels */
    active[0][0] = 0;
    active[1][0] = 0;
    active[0][1] = 0;
    active[1][1] = 0;

    for (chan = 0; chan < nchan; chan++)
	CHECK(e1432_set_active(hw, chan_list[chan],
			       active[0][chan] ?
			       E1432_CHANNEL_ON : E1432_CHANNEL_OFF));

    return 0;
}

/*ARGSUSED*/
static int
setup2(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
       int fft, int active[][NCHAN_MAX])
{
    int type, chan;

    /* Initialize active array */
    for (type = 0; type < 2; type++)
	for (chan = 0; chan < nchan; chan++)
	{
	    if (type > 0 && !fft)
		active[type][chan] = 0;
	    else
		active[type][chan] = 1;
	}

    /* All channels active */

    for (chan = 0; chan < nchan; chan++)
	CHECK(e1432_set_active(hw, chan_list[chan],
			       active[0][chan] ?
			       E1432_CHANNEL_ON : E1432_CHANNEL_OFF));

    return 0;
}

/*ARGSUSED*/
static int
setup3(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
       int fft, int active[][NCHAN_MAX])
{
    int type, chan;

    /* Initialize active array */
    for (type = 0; type < 2; type++)
	for (chan = 0; chan < nchan; chan++)
	{
	    if (type > 0 && !fft)
		active[type][chan] = 0;
	    else
		active[type][chan] = 1;
	}

    /* Deactivate third channel */
    active[0][2] = 0;
    active[1][2] = 0;

    for (chan = 0; chan < nchan; chan++)
	CHECK(e1432_set_active(hw, chan_list[chan],
			       active[0][chan] ?
			       E1432_CHANNEL_ON : E1432_CHANNEL_OFF));

    return 0;
}

/*ARGSUSED*/
static int
setup4(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
       int fft, int active[][NCHAN_MAX])
{
    int type, chan;

    /* Initialize active array to no channels active */
    for (type = 0; type < 2; type++)
	for (chan = 0; chan < nchan; chan++)
	    active[type][chan] = 0;

    /* Activate first channel */
    active[0][0] = 1;
    active[1][0] = fft ? 1 : 0;

    for (chan = 0; chan < nchan; chan++)
	CHECK(e1432_set_active(hw, chan_list[chan],
			       active[0][chan] ?
			       E1432_CHANNEL_ON : E1432_CHANNEL_OFF));

    return 0;
}

static int
check_trailer(struct e1432_trailer *trailer, FLOATSIZ32 clock_freq,
	      double span, int chan, int type)
{
    double  tmp;
    int     df2, df5;

    if (trailer->zoom_corr != 0)
    {
	/* Zoom correction is not currently implemented */
	(void) fprintf(stderr,
		       "%s: trailer zoom corr non-zero: %g (0x%lx)\n",
		       progname, trailer->zoom_corr,
		       *(long *) &trailer->zoom_corr);
	return -1;
    }
    if (trailer->gap < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer gap negative: 0x%lx\n",
		       progname, trailer->gap);
	return -1;
    }
    if (trailer->rpm1 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm1 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm1,
		       *(long *) &trailer->rpm1);
	return -1;
    }
    if (trailer->rpm2 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm2 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm2,
		       *(long *) &trailer->rpm2);
	return -1;
    }
    if (trailer->peak != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer peak non-zero: %g (0x%lx)\n",
		       progname, trailer->peak,
		       *(long *) &trailer->peak);
	return -1;
    }
    if (trailer->rms != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rms non-zero: %g (0x%lx)\n",
		       progname, trailer->rms,
		       *(long *) &trailer->rms);
	return -1;
    }

    /* Compute df2 and df5 from clock_freq and span */
    tmp = span * 2.56;
    df2 = 0;
    df5 = 0;
    while (tmp < clock_freq * 0.9999)
    {
	df2++;
	tmp *= 2;
    }
    if (tmp > clock_freq * 1.0001)
    {
	tmp /= 8;
	tmp *= 5;
	df2 -= 3;
	df5++;
	if (tmp > clock_freq * 1.0001 || tmp < clock_freq * 0.9999)
	{
	    (void) fprintf(stderr,
			   "%s: invalid span/clock_freq combination: %g/%g\n",
			   progname, span, clock_freq);
	    return -1;
	}
    }

    if (df2 != ((trailer->info & E1432_TRAILER_INFO_DEC_2_MASK)
		>> E1432_TRAILER_INFO_DEC_2_SHIFT))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df2 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df2);
	return -1;
    }
    if (df5 != ((trailer->info & E1432_TRAILER_INFO_DEC_5) != 0))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df5 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df5);
	return -1;
    }

    if (((trailer->info & E1432_TRAILER_INFO_CHAN_MASK) >>
	 E1432_TRAILER_INFO_CHAN_SHIFT) != chan - 1)
    {
	(void) fprintf(stderr,
		       "%s: trailer info chan mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, chan - 1);
	return -1;
    }
    if (((trailer->info & E1432_TRAILER_INFO_TYPE_MASK) >>
	 E1432_TRAILER_INFO_TYPE_SHIFT) != type)
    {
	(void) fprintf(stderr,
		       "%s: trailer info type mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, type);
	return -1;
    }

    return 0;
}

static int
wait_block_avail(E1432ID hw, int group, int scan, int verbose,
		 long blocksize, double span)
{
    clock_t start, timeout;
    int     status;

    timeout = (2 + 2 * (blocksize / (span * 2.56))) * CLOCKS_PER_SEC;
    if (verbose > 2)
	(void) printf("Waiting %g sec for block available\n",
		      (double) timeout / CLOCKS_PER_SEC);
    start = clock();
    while ((status = e1432_block_available(hw, group)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_block_available(hw, group)) == 0)
	{
	    (void) fprintf(stderr, "%s: e1432_block_available: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_block_available: returned %d\n",
		       progname, status);
	return -1;
    }
    if (verbose > 0)
	(void) printf("Scan %d block available found\n", scan);

    return 0;
}

#ifdef	PLOT
/* Open plot windows */
static int
open_windows(int nchan_plot, SHORTSIZ16 chanlist[], char *id[],
	     FLOATSIZ32 *dataBuffer, double span,
	     long blocksize, double yscale)
{
    Display *disp;
    char    geometry[32], title[8];
    int     divisor, chan;
    int     xscreen, yscreen, xwidth, ywidth, xstart, ystart;

    /* Get screen size in pixels */
    xscreen = 0;
    yscreen = 0;
    disp = XOpenDisplay("");
    if (disp != 0)
    {
	xscreen = XDisplayWidth(disp, 0);
	yscreen = XDisplayHeight(disp, 0);
	(void) XCloseDisplay(disp);
    }
    if (xscreen == 0 || yscreen == 0)
    {
	/* If anything failed, guess at a size */
	xscreen = 1024;
	yscreen = 768;
    }

    divisor = 2;	/* Maximum window size 1/2 screen dimension */
    while ((divisor * divisor) < nchan_plot)
	divisor++;

    /* Window size in pixels */
    xwidth = xscreen / divisor - X_SPACING;
    ywidth = yscreen / divisor - Y_SPACING;

    for (chan = 0; chan < nchan_plot; chan++)
    {
	/* Window location in pixels */
	xstart = chan % divisor * (xwidth + X_SPACING) + 2;
	ystart = chan / divisor * (ywidth + Y_SPACING) + 3;

	(void) sprintf(geometry, "%dx%d+%d+%d",
		       xwidth, ywidth, xstart, ystart);
	(void) sprintf(title, "%3d", chanlist[chan]);

	id[chan] = xplot_init_plot(dataBuffer + chan * blocksize,
				   blocksize, span,
				   yscale, -yscale, TIME_TRACE,
				   geometry, title);
    }

    return 0;
}
#endif

/*ARGSUSED*/
static int
run_meas(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
	 int active[][NCHAN_MAX], double *signal_list,
	 double clock_freq, double span, int verbose)
{
#ifndef	PLOT
    FLOATSIZ32 buffer[BLOCKSIZE * NCHAN_MAX];
#endif
    LONGSIZ32 count;
    struct e1432_trailer trailer;
    double  rms;
    int     scan, chan, type, i;

#ifdef	PLOT
    memset(buffer, BLOCKSIZE * NCHAN_MAX * sizeof buffer[0], 0);
#endif

    CHECK(e1432_init_measure(hw, group));

    for (scan = 0; scan < SCANS_PER_MEAS; scan++)
    {
	/* Wait for block available */
	if (wait_block_avail(hw, group, scan, verbose,
			     BLOCKSIZE, span) < 0)
	    return -1;

	/* Read the data */
	for (type = 0; type < 2; type++)
	    for (chan = 0; chan < nchan; chan++)
	    {
		if (!active[type][chan])
		    continue;

		if (verbose > 1)
		    (void) printf("Reading chan %d type %d\n",
				  chan, type);

		CHECK(e1432_read_float32_data(hw, chan_list[chan],
					      type == 0 ?
					      E1432_TIME_DATA :
					      E1432_FREQ_DATA,
					      buffer + BLOCKSIZE * chan,
					      BLOCKSIZE, &trailer,
					      &count));
		if (count != BLOCKSIZE)
		{
		    (void) fprintf(stderr,
				   "%s: e1432_read_float64_data: "
				   "actual count was %ld\n",
				   progname, count);
		    return -1;
		}

#ifdef	PLOT
		if (type == 0)
		{
		    xplot_data_update(plotID[chan]);
		    xplot_check_events(plotID[chan]);
		    if (chan == 0)
			(void) sleep(1);
		}
#endif

		if (check_trailer(&trailer, clock_freq, span,
				  chan_list[chan], type) < 0)
		    return -1;

		if (signal_list[chan] >= 0)
		{
		    /* Calculate RMS level */
		    rms = 0;
		    for (i = 0; i < BLOCKSIZE; i++)
			rms += buffer[BLOCKSIZE * chan + i] *
			    buffer[BLOCKSIZE * chan + i];
		    if (type == 0)
			rms /= BLOCKSIZE;
		    else
			/* No divide by blocksize for freq data, but
			   scale down by two because freq data was
			   peak rather than RMS. */
			rms /= 2;
		    rms = sqrt(rms);

		    /* The 0.1 limit is arbitrary */
		    if (fabs(rms - signal_list[chan]) > 0.1)
		    {
			(void) fprintf(stderr,
				       "%s: chan %d: RMS %g, expected %g\n",
				       progname, chan, rms, signal_list[chan]);
			return -1;
		    }
		}
	    }
    }

#ifdef	PLOT
    (void) sleep(1);
#endif

    return 0;
}

static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
    int active[][NCHAN_MAX], double *signal_list,
    double clock_freq, double span, int verbose)
{
    int     i, chan;

    if (verbose > 0)
	(void) printf("Running at clock freq %g, span %g\n",
		      clock_freq, span);

    for (i = 0; i < LOOPS; i++)
    {
	CHECK(run_meas(hw, group, nchan, chan_list, active,
		       signal_list, clock_freq, span, verbose));
	/* One specific problem we has was with turning channels off and
	   then on and restarting a measurement.  This sometimes
	   caused channel swap. */
	if (verbose > 0)
	    (void) printf("Trying active off/on\n");
	CHECK(e1432_set_active(hw, group, E1432_CHANNEL_OFF));
	for (chan = 0; chan < nchan; chan++)
	    CHECK(e1432_set_active(hw, chan_list[chan],
				   active[0][chan] ?
				   E1432_CHANNEL_ON : E1432_CHANNEL_OFF));
	/*CHECK(e1432_set_active(hw, group, E1432_CHANNEL_ON));*/
	CHECK(run_meas(hw, group, nchan, chan_list, active,
		       signal_list, clock_freq, span, verbose));
	/* Perhaps auto zero can affect channel swap */
	if (verbose > 0)
	    (void) printf("Trying auto zero\n");
	CHECK(e1432_auto_zero(hw, group));
	CHECK(run_meas(hw, group, nchan, chan_list, active,
		       signal_list, clock_freq, span, verbose));
	/* Perhaps auto range can affect channel swap */
	if (verbose > 0)
	    (void) printf("Trying auto range\n");
	CHECK(e1432_auto_range(hw, group, 0));
	CHECK(run_meas(hw, group, nchan, chan_list, active,
		       signal_list, clock_freq, span, verbose));
	/* Perhaps finish measure can affect channel swap */
	if (verbose > 0)
	    (void) printf("Trying finish measure\n");
	CHECK(e1432_finish_measure(hw, group));
	CHECK(run_meas(hw, group, nchan, chan_list, active,
		       signal_list, clock_freq, span, verbose));
    }

    return 0;
}

static int
run2(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
     int active[][NCHAN_MAX], double *signal_list,
     double clock_freq, double span, int fft, int verbose)
{
    /* Only 1 channel */
    if (verbose > 0)
	(void) printf("Running with 1 channel active - %g/%g\n",
		      clock_freq, span);
    CHECK(e1432_preset(hw, group));
    CHECK(setup(hw, group, nchan, chan_list, fft, clock_freq, span));
    CHECK(setup4(hw, group, nchan, chan_list, fft, active));
    CHECK(run_meas(hw, group, nchan, chan_list, active, signal_list,
		   clock_freq, span, verbose));

    /* All channels */
    if (verbose > 0)
	(void) printf("Running with all channels active - %g/%g\n",
		      clock_freq, span);
    CHECK(e1432_preset(hw, group));
    CHECK(setup(hw, group, nchan, chan_list, fft, clock_freq, span));
    CHECK(setup2(hw, group, nchan, chan_list, fft, active));
    CHECK(run_meas(hw, group, nchan, chan_list, active, signal_list,
		   clock_freq, span, verbose));

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-fuvV] [-L laddr] [-n nchan] [-N nmod]\n"
		   "\t[-x lev[,lev]]\n"
		   "\t-f: Set up for fft data\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n"
		   "\t-x: Specify expected signal levels\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    double  signal_list[NCHAN_MAX];
    char   *p;
    int     active[2][NCHAN_MAX];
    int     c, i, fft, nmod, verbose, group, nchan, chan, offset, count;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    fft = 0;
    laddr[0] = 8;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    verbose = 0;
    for (chan = 0; chan < NCHAN_MAX; chan++)
	signal_list[chan] = -1;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "fL:n:N:uvVx:")) != -1)
	switch (c)
	{
	case 'f':
	    fft = 1;
	    break;
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'x':
	    offset = 0;
	    for (chan = 0; chan < NCHAN_MAX; chan++)
	    {
		if (sscanf(optarg + offset, "%lg%n",
			   &signal_list[chan], &count) != 1)
		{
		    if (strlen(optarg + offset) > 0)
		    {
			(void) fprintf(stderr,
				       "%s: invalid expected signal level: '%s'\n",
				       progname, optarg);
			usage();
		    }
		    break;
		}
		offset += count;
		count = 0;
		(void) sscanf(optarg + offset, " ,%n", &count);
		offset += count;
	    }
	    break;
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Run the measurement */
    if (init(nmod, laddr, &hw, &group, &nchan, chan_list) < 0)
	return EXIT_FAILURE;

#ifdef	PLOT
    if (open_windows(nchan, chan_list, plotID, buffer,
		     20000, BLOCKSIZE, RANGE) < 0)
	return EXIT_FAILURE;
#endif

    for (i = 0; i < LOOPS; i++)
	if (run2(hw, group, nchan, chan_list, active, signal_list,
		 51200, 20000, fft, verbose) < 0)
	    return EXIT_FAILURE;
    for (i = 0; i < LOOPS; i++)
	if (run2(hw, group, nchan, chan_list, active, signal_list,
		 65536, 12800, fft, verbose) < 0)
	    return EXIT_FAILURE;

    if (setup(hw, group, nchan, chan_list, fft, 51200, 20000) < 0)
	return EXIT_FAILURE;
    if (setup1(hw, group, nchan, chan_list, fft, active) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, active, signal_list,
	    51200, 20000, verbose) < 0)
	return EXIT_FAILURE;

    if (setup(hw, group, nchan, chan_list, fft, 65536, 12800) < 0)
	return EXIT_FAILURE;
    if (setup2(hw, group, nchan, chan_list, fft, active) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, active, signal_list,
	    65536, 12800, verbose) < 0)
	return EXIT_FAILURE;

    if (setup(hw, group, nchan, chan_list, fft, 40960, 16000) < 0)
	return EXIT_FAILURE;
    if (setup3(hw, group, nchan, chan_list, fft, active) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, active, signal_list,
	    40960, 16000, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
